 # In Text Object

## Introduction

In order to fully describe a substring, a segment requires a reference string.  In other words, both a ``Span`` and ``str`` are required.  In Pawpaw, this is represented in via the *In Text Object*[^ito_name], implemented as class ``Ito``.

## Namespace

``Ito`` is as a root level object in the namespace ``pawpaw``::

```python
>>> from pawpaw import Ito
>>> Ito.__name__
'Ito'
```

## Creation

A large number of techniques are available for creating ``Ito`` instances:

* From a ``str`` or another ``Ito`` via ``.__init__``
* Static ``.from_...`` helper methods, e.g., ``.from_substrings``, ``.from_match``, ``.__init__``, etc.
* ``Ito.clone`` method
* ``Ito`` slice operation
* Return value of methods, e.g. ``.split``, ``.str_partition``, ``.str_strip``, etc.

### ``.__init__``

#### Creating from a ``str``

Instantiating an ``Ito`` against a reference ``str`` is straightforward:

```python
>>> from pawpaw import Ito
>>> s = 'Hello, World!'
Ito('Hello, World!', 0, 13, None)
```

The ``str`` value supplied to the constructor is accessible via a ``.string`` property.  The constructor features optional ``start`` and ``stop`` parameters, which are are *Python-style indices*.  The values for ``str``, ``start``, and ``stop`` are used to create a ``Span``, which is accessible via the ``.span`` property:

```python
>>> i = Ito('abcd', 1, -1)
>>> i.span
Span(start=1, stop=3)
```

Having separate ``start`` and ``stop`` parameters is more idiomatic with Python than using a single, ``Span`` parameter.  However, given a span you can easily supply these values via unpacking:

```python
>>> from pawpaw import Span
>>> span = Span(1, 3)
>>> i = Ito('abcd', *span)
>>> i
Ito('abcd', 1, 3, None)
```

The ``start`` and ``stop`` values for the span are accessible via ``.start`` and ``.stop`` properties on *both* the ``Ito`` and ``Span``:

```python
>>> i = Ito('abc')
>>> i.span == (i.start, i.stop)
True
```

``Ito`` also features a ``.desc`` property[^desc_name], which serves as a tag for describing the *type* of segment referred to.  An optional ``desc`` is the fourth and final parameter to the constructor:

```python
>>> s = 'John Doe'
>>> i1 = Ito(s, stop=4, desc='FN')
>>> i2 = Ito(s, 5, desc='LN')
>>> f'{i1.desc}: "{i1}";  {i2.desc}: "{i2}"'
'FN: "John"; LN: "Doe"'
```

The ``.string``, ``.start``, ``.stop``, and ``.span`` properties are all read-only and invariant.  Only the ``.desc`` property can be changed post-instantiation.  This is by design and ensures that substrings described by itos are immutable[^str_immutable] in Pawpaw.  Trying to set any of these values results in an Error:

```python
>>> i = Ito('abc')
>>> i.string = 'xyz'
Traceback (most recent call last):
  File "<stdin>", line 1, in <module>
Attribute Error: can't set attribute 'string'
```

<!--
.. admonition:: Key Concept

   ``Ito`` strings and spans are immutable
-->

#### Creating from another ``Ito``

Frequently, you'll want to create one substring from another.  This can be achieved by supplying an ``Ito`` as the first parameter to the constructor instead of a ``str``.  The ``start`` and ``stop`` parameters are *relative to the first parameter*[^src_param]:

```python
>>> s = '_abc_'
>>> i1 = Ito(s, 1, -1)   # str basis; skip first & last chars of str s
>>> i2 = Ito(i1, 1, -1)  # Ito basis; skip first and last chars of Ito i1
>>> str(i1), str(i2)
('abc', 'b')
```

### ``.from_match_group``

The ``.from_match_group`` method allows you to create ``Ito`` instances from an ``re.Match`` object and group key.  The ``group`` parameter accepts a group key, whose type can by either an ``int`` or ``str``, and which defaults to 0 (the group key corresponding to the *entire* match):

```python
>>> s = 'John doe'
>>> re = regex.compile(r'(?P<fn>.+)\s(?P<ln>.+)')
>>> m = re.fullmatch(s)
>>> str(Ito.from_match_group(m))  # group defaults to 0 (the entire match)
'John Doe'
>>> str(Ito.from_match_group(m, 2))  # group key 2
'Doe'
>>> str(Ito.from_match_group(m, 'fn'))  # group key 'fn'
'John'
```

An optional ``desc`` parameter accepts either a:

1. ``str``
2. a method inputs are an ``re.Match`` and group key, and whose output is a ``str``

If omitted, the ``desc`` parameter defaults to:

```python
lambda m, gk: str(gk)
```

which returns a string corresponding to the supplied ``group`` parameter.

Example:

```python
>>> Ito.from_match_group(m, 'fn')  # desc defaults to group parameter ('fn')
Ito('John Doe', 0, 4, 'fn')
>>> Ito.from_match_group(m, 'fn', lambda mm, gk: f'{len(mm)}: {gk}')
Ito('John Doe', 0, 4, '3: fn')
```

### ``.from_match``

The ``.from_match`` method allows you to create a fully hierarchical ``Ito`` instance whoses nodes correspond to the captures of an ``re.Match`` object.  Like the ``.from_match_group`` method, a ``desc`` parameter can be supplied, and which defaults to the same lambda.  Group keys are dynamically passed to this lambda to yield descriptors.  For named captures, the ``str`` group key (i.e., the capture name) is supplied, otherwise, the ``int`` group key is used.

```python
>>> s = 'John doe'
>>> re = regex.compile(r'(?P<fn>.+)\s(?P<ln>.+)')
>>> m = re.fullmatch(s)
>>> i = str(Ito.from_match(m))
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> print(tree_vis.dumps(i))
(0, 8) '0' : 'John Doe'
├──(0, 4) 'fn' : 'John'
└──(5, 8) 'ln' : 'Doe'
```

A variable length argument, ``exclude_keys``, can be used to indicate which groups should be excluded from the result.  Note that a ``ValueError`` is raised if you include zero in ``exclude_keys``.  This is because ``.from_match`` returns a single Ito whose root is the entire-match.  Excluding the zero-match could potentially result in a return value consiting of more than one tree, and so it is not allowed.

```python
>>> i = str(Ito.from_match(m, 'fn'))
>>> print(tree_vis.dumps(i))
(0, 8) '0' : 'John Doe'
└──(5, 8) 'ln' : 'Doe'
```

### ``.from_re``

The ``.from_re`` method allows you to create ``Ito`` instances from a ``regex.Pattern``.  The regex method ``.finditer`` is called, and each match is handled as if you called ``Ito.from_match``:

```python
>>> s = 'A1 B2 C3'
>>> re = regex.compile(r'(?<letter>[A-Z])(?<digit>\d)')
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> for i in Ito.from_re(re, s):
>>>   print(tree_vis.dumps(i))
(0, 2) '0' : 'A1'
├──(0, 1) 'letter' : 'A'
└──(1, 2) 'digit' : '1'

(3, 5) '0' : 'B2'
├──(3, 4) 'letter' : 'B'
└──(4, 5) 'digit' : '2'

(6, 8) '0' : 'C3'
├──(6, 7) 'letter' : 'C'
└──(7, 8) 'digit' : '3'
```

### ``.from_spans``

The ``.from_spans`` method allows you to create a sequence of ``Ito`` instances from one or more spans.  An optional ``desc`` parameter, if provided, is used as the descriptor for the generated objects:

```python
>>> s = 'John doe'
>>> for i in Ito.from_spans(s, pawpaw.Span(0, 2), pawpaw.Span(3, 6), desc='my_desc'):
>>>     print(f'{i:%span: %desc: "%substr"}')
(0, 2) my_desc: "Jo"
(3, 6) my_desc: "n D"
```

### ``.from_gaps``

The ``.from_gaps`` method allows you to create a sequence of ``Ito`` instances by providing spans you want *excluded* from the result.  An optional ``desc`` parameter, if provided, is used as the descriptor for the generated objects:

```python
>>> s = 'John doe'
>>> for i in Ito.from_gaps(s, pawpaw.Span(0, 2), pawpaw.Span(3, 6), desc='my_desc'):
>>>     print(f'{i:%span: %desc: "%substr"}')
(2, 3) my_desc: "h"
(6, 8) my_desc: "oe"
```

### ``.from_substrings``

Given a sequence of substrings, you can create an ``Ito`` sequence using the static ``.from_substrings`` method.  When supplied with a ``src`` of type ``str`` or ``Ito`` and one or more non-overlapping substrings, this method generates a sequence of ``Ito`` objects:

```python
>>> s = 'A B C'
>>> [*Ito.from_substrings(s, *s.split())]
[Ito('A B C', 0, 1, None), Ito('A B C', 2, 3, None), Ito('A B C', 4, 5, None)]
```

### ``.clone``

The ``Ito.clone`` method allows you to duplicate existing ``Ito`` instances.  Optional parameters ``start``, ``stop``, and ``desc`` allow you to supply alternate values for the resulting object.  However, unlike in the ``Ito`` constructor, in ``.clone`` the parameters ``start`` and ``stop`` are *relative to the underlying string*.

Unlike ``.__init__``, the ``.clone`` preserves the:

* ``.value``
* ``.children``
* *type* of Ito (i.e., derived)

in the result.  An optional ``clone_children`` parameter (defaults to True) can be provided if you do _not_ wish to clone the children.

```python
>>> s = 'John doe'
>>> i = Ito(s)
>>> i.children.add(*i.str_split())
>>> j = i.clone()
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> print(tree_vis.dumps(j))
(0, 8) 'None' : 'John Doe'
├──(0, 4) 'None' : 'John'
└──(5, 8) 'None' : 'Doe'
```

## Usage

### Substring access

There are many ways to access the substring referred to by an ``Ito``\ ::

```python
>>> i = Ito('abc')
>>> i.__str__()  # direct call to .__str__
'abc'
>>> str(i)       # indirect
'abc'
>>> f'{i}'       # indirect
'abc'
```

Note that Pawpaw supports Ito composite formatting for python format strings.  See [Visualization](./7.%20Visualization.md) in the docs for more detail.

### ``.value`` method

The ``.value`` method allows you to define runtime and/or polymorphic value extraction for the substring referenced by ``Ito``.  Its default behavior is to defer to ``.__str__()``.

#### Polymorphic ``.value()`` implementation

An ``Ito`` subclass can override the default behavior of ``.value()``.  For example:

```python
class IntIto(Ito)
    def value(self) -> Typing.Any:
        return int(str(self))
```

You can use it as follows:

```python
>>> s = '1 2 3 4'
>>> itos = [*Ito.from_substrings(s, *s.split())]
>>> [i.value() for i in itos]  # default behavior, resolves to str
['1', '2', '3']
>>> itos = [*IntIto.from_substrings(s, *s.split())]
>>> [i.value() for i in itos]  # custom behavior, resolves to int
[1, 2, 3]
```

#### Runtime ``.value()`` implementation

``Ito`` provides a .value_func property accessors that allow for easy, run-time assignment of a custom value method without relying on strict inheritance:

```python
>>> s = '1'
>>> i = Ito(s)
>>> i.value()
'1'
>>> i.value_func = lambda ito: int(str(ito))
>>> i.value()
1
>>> i.value_func = None  # resets to default behavior
>>> i.value()
'1'
```

Runtime-set ``.value`` methods *are* preserved via cloning:

```python
>>> s = '12345'
>>> i = Ito(s)
>>> i.value_func = lambda ito: int(str(ito))
>>> i.value()
12345
>>> j = i.clone(stop=3)
>>> str(j)
'123'
>>> j.value()
123
```

## Equivalence Methods


Accessing the substring identified by an ``Ito`` requires first calling the ``.__str__()`` method.  However, this introduces an inefficiency - namely, a new string must be allocated and initialized in memory.  This is unavoidable if you need an actual ``str``, however, often the substring is only needed as the basis for another operation.

For example, let's say you want to call ``str.find`` for some value on the substring.  This method's signature includes parameters for start and stop indices, which if used, bypasses the need to first create a substring:

```python
>>> s = ' abc '
>>> i = Ito(s, 1, -1)
>>> str(i).find('b')  # Inefficient...
1
>>> i.string.find('b', i.start, i.stop)  # Better...
1
```

Although better, the resulting syntax is more lengthy, and is easily overlooked.  To encourage this type of efficiency, ``Ito`` features are large number of helper  methods for various ``str`` and ``regex`` methods, prefixed with ``str_`` and ``regex_`` respectively.  For example ``str.find`` can be performed using ``Ito.str_find`` as follows:

```python
>>> i.str_find('b')  # Best!
1
```

These methods:

* have the same name as their ``str`` or ``regex`` counterpart, prefixed with ``str_`` and ``regex_`` respectively.
* have identical parameters
* have return values that:
  * are identical if the the counterpart method returns a non-``str``
  * are an ``Ito`` when the counterpart method returns a ``str``
* have no effects on the ``Ito`` used to call them
* are available for associated methods that are *non-modifying* or *return values that are themselves substrings for the basis string*.  For example, ``str.count``, ``str.endswith``, and ``str.islower`` have equivalence methods, while ``str.upper()`` does not.
 
### str equivalence methods

| ``str`` method | ``Ito`` equivalence method |
|:--------------:|:--------------------------:| 
|     count      |         str_count          |
|    endswith    |        str_endswith        |
|   startswith   |       str_startswith       |
|      find      |          str_find          |
|     index      |         str_index          |
|     rfind      |         str_rfind          |
|     rindex     |         str_rindex         |
|    isalnum     |        str_isalnum         | 
|    isalpha     |        str_isalpha         |
|    isascii     |        str_isascii         |
|   isdecimal    |       str_isdecimal        |
|    isdigit     |        str_isdigit         |
|  isidentifer   |      str_isidentifer       |
|   isislower    |       str_isislower        |
|   isnumeric    |       str_isnumeric        |
|  isprintable   |      str_isprintable       |
|    isspace     |        str_isspace         |
|    istitle     |        str_istitle         |
|    isupper     |        str_isupper         |
|     lstrip     |         str_lstrip         |
|     rstrip     |         str_rstrip         |
|     strip      |         str_strip          |
|   partition    |       str_partition        |
|   rpartition   |       str_rpartition       |
|     rsplit     |         str_rsplit         |
|     split      |         str_split          |
|   splitlines   |       str_splitlines       |
|  removeprefix  |      str_removeprefix      |
|  removesuffix  |      str_removesuffix      |

### regex equivalence methods

| ``regex`` method | ``Ito`` equivalence method |
|:----------------:|:--------------------------:| 
|      search      |        regex_search        |
|      match       |        regex_match         |
|    fullmatch     |      regex_fullmatch       |
|      split       |        regex_split         |
|    splititer     |      regex_splititer       |
|     findall      |       regex_findall        |
|     finditer     |       regex_finditer       |

## ``.children`` 


An ``Ito`` is fully hierarchical through its ``.parent`` and ``.children`` properties.  This allows an ``Ito`` to both represent a segment and act as a node within a graph[^is_tree_graph].

<!--
.. admonition:: Key Concept

   Given a set of rules that identify segment lineages, an ``Ito`` hierarchy can be used as topologically complete data collection for all discovered segments.
-->

   The ``.desc`` property allows for users to define the *type* of segment an ``Ito`` refers to.  For example, a set of NLP rules might create segments of type *paragraph*, *sentence*, *word*, etc.  Conversely, rules for XML segmentation have segments named *element*, *attribute*, *key*, *value*, etc.

<!--
.. admonition:: Key Concept

   Segment relevance is determined by both a) the location of ``Ito`` within its hierarchy and b) the .desc for the ``Ito``
-->

An ``Ito`` is not an arbitrary data collection.  Rather, it defines segments within text and can have ancestors, siblings, and descendants.  This results in several implicit requirements:

* Children are contained in their parent: The ``.span`` for a child ``Ito`` is contained within the ``.span`` of its ``.parent``
* Non-overlapping siblings: An ``Ito`` does not overlap any other ``Ito`` objects that share the same ``.parent``
* Ordered siblings: Siblings are ordered within their parent by their ``.span.start``; because they are non-overlapping, the ``.span.stop`` for an ``Ito`` will always be less than or equal to the ``.span.start`` of the subsequent sibling

### Adding children

#### ``.add``

...adds children to parent...

#### ``.add_hierarchical``

...adds descendants to parent...

[^ito_name]: The name "In Test Object" is historical, and dates back to earlier projects I developed.  I've chosen to keep this name because "Ito" makes for a short, convenient type name!

[^desc_name]: In earlier versions of the framework, this was named ``descriptor``.  Its usage, however, is frequent, and a ten-character long identifier makes for more verbose and less readable code.

[^str_immutable]: In Python, strings are also immutable.

[^src_param]: This is why the first parameter is named ``src`` and not ``string``, indicating that *multiple* types can be used as the source for your substring

[^is_tree_graph]: Because a child ``Ito`` must be equal to or contained by a parent ``Ito``, this is a *tree* graph.
